function [alphae, alphah, alphaeh] = polarizabilities(epsin, epsout, muin,...
    muout, chi, a, kw, varargin)

% This function obtains the polarizabilities tensors for dipoles ocuppying
% a cubic unit cell of side a. It's designed for scatterers that are
% composed of a material with a certain electric permittivity, magnetic
% permeability and mixed magnetic-electric susceptibility.

%%%%%% INPUT ARGUMENTS %%%%%%
%%% epsin:  electric permittivity of the scatterer         %%%
%%% epsout: electric permittivity of the surrounding media %%%
%%% muin:   magnetic permeability of the scatterer         %%%
%%% theta:  magnetic permeability of the surrounding media %%%
%%% chi:    mixed magnetic electric susceptibility         %%%
%%% a:      side length of the cubic unit cell             %%%
%%% kw:     exciting wavenumber                            %%%
%%% Ns:     number of scatterers                           %%%


%%%%%% OUTPUT ARGUMENTS %%%%%%
%%% alphae:  electric polarizability of the unit cell                %%% 
%%% alphah:  magnetic polarizability of the unit cell                %%%                    
%%% alphaeh: mixed electric-magnetic polarizability of the unit cell %%%

% Constants %
load('vacPer.mat','eps0');

% Predefinition of the polarizabilities for loop %
K  = length(kw);
Ns = length(epsin(1,:));
alphae  = zeros(K,Ns,3,3);
alphah  = alphae;
alphaeh = alphae;

% Adapting posible constant permitivities to the frequency loop %
epsin  = (epsin.'*ones(1,K)).';
epsout = (epsout.'*ones(1,K)).'; 
chi    = (chi.'*ones(1,K)).';


% Applaying the Clausius-Mossoti relation for each frequency and material %
for j = 1:Ns
    for i = 1:K

        alphae(i,j,:,:)  =  4*pi*eps0*epsout(i)*cm_polarizability(epsin(i,j),epsout(i),a(j),kw(i));
        alphah(i,j,:,:)  =  cm_polarizability(muin,muout,a(j),kw(i));
        alphaeh(i,j,:,:) = -3*a(j)^3*(chi(i,j)/120/pi)*sqrt(epsout(i))/(epsin(i,j)+2)*eye(3);

    end
end


if(~isempty(varargin))
    Nd = varargin{1};
    [alphae,alphah,alphaeh] = joinScatterers(alphae, alphah, alphaeh, Nd);
end


% Obviating unnesesary dimensions %
alphae  = squeeze(alphae);
alphah  = squeeze(alphah);
alphaeh = squeeze(alphaeh);


end
